<?php
/**
 * Plugin Name: KDDMantenimiento
 * Plugin URI: https://kddbusiness.com
 * Description: Plugin de mantenimiento, muestra una página de mantenimiento personalizable con imagen/video de fondo (YouTube y Vimeo) y logo ajustable para KDDClub.
 * Version: 2.0
 * Author: KDD Business
 * License: GPL v2 or later
 * Text Domain: kddmantenimiento
 * Domain Path: /languages
 */

// Evitar acceso directo
if (!defined('ABSPATH')) {
    exit;
}

class KDDMantenimiento {
    
    private $options;
    
    public function __construct() {
        $this->options = get_option('kdd_mantenimiento_settings');
        add_action('init', array($this, 'init'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'settings_init'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // Hook de activación
        register_activation_hook(__FILE__, array($this, 'activate'));
        
        // Inicializar componentes
        $this->load_includes();
    }
    
    private function load_includes() {
        require_once plugin_dir_path(__FILE__) . 'includes/social-media.php';
        require_once plugin_dir_path(__FILE__) . 'includes/whatsapp.php';
        require_once plugin_dir_path(__FILE__) . 'includes/text-editor.php';
        
        // Inicializar módulos
        $this->social_media = new KDDMantenimiento_Social_Media($this->options);
        $this->whatsapp = new KDDMantenimiento_WhatsApp($this->options);
        $this->text_editor = new KDDMantenimiento_Text_Editor();
    }
    
    public function activate() {
        // Configuración por defecto al activar
        $default_settings = array(
            'enable_maintenance' => '0',
            'background_type' => 'image',
            'background_color' => '#2c3e50',
            'logo_size' => '200px',
            'custom_message' => 'Estamos realizando tareas de mantenimiento para mejorar tu experiencia. Volveremos pronto.',
            'text_color' => '#ffffff',
            'video_type' => 'youtube',
            'mobile_fallback' => '1',
            // Nuevos campos para v2.0
            'social_instagram' => '',
            'social_tiktok' => '',
            'social_facebook' => '',
            'social_twitter' => '',
            'whatsapp_number' => '',
            'whatsapp_enabled' => '0'
        );
        
        if (false === get_option('kdd_mantenimiento_settings')) {
            add_option('kdd_mantenimiento_settings', $default_settings);
        }
        
        // Actualizar versión si es necesario
        update_option('kddm_version', '2.0');
    }
    
    public function enqueue_admin_scripts($hook) {
        if ('settings_page_kdd_mantenimiento' !== $hook) {
            return;
        }
        
        wp_enqueue_media();
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('kdd-admin-script', plugin_dir_url(__FILE__) . 'assets/js/admin.js', array('jquery', 'wp-color-picker'), '2.0', true);
        wp_enqueue_style('kdd-admin-style', plugin_dir_url(__FILE__) . 'assets/css/admin.css', array(), '2.0');
        
        // Cargar editor de texto enriquecido
        wp_enqueue_editor();
    }
    
    public function init() {
        load_plugin_textdomain('kddmantenimiento', false, dirname(plugin_basename(__FILE__)) . '/languages');
        
        // Mostrar página de mantenimiento si está activo y el usuario no es admin
        if (!current_user_can('manage_options') && !is_admin() && !wp_doing_ajax()) {
            if (isset($this->options['enable_maintenance']) && $this->options['enable_maintenance'] == '1') {
                $this->show_maintenance_page();
            }
        }
    }
    
    public function add_admin_menu() {
        add_options_page(
            'KDD Mantenimiento',
            'KDD Mantenimiento',
            'manage_options',
            'kdd_mantenimiento',
            array($this, 'options_page')
        );
    }
    
    public function settings_init() {
        register_setting('kdd_mantenimiento', 'kdd_mantenimiento_settings', array($this, 'sanitize_settings'));
        
        // Sección principal
        add_settings_section(
            'kdd_mantenimiento_section',
            'Configuración del Modo Mantenimiento',
            array($this, 'settings_section_callback'),
            'kdd_mantenimiento'
        );
        
        // Campos de configuración principales
        add_settings_field(
            'enable_maintenance',
            'Activar Modo Mantenimiento',
            array($this, 'enable_maintenance_render'),
            'kdd_mantenimiento',
            'kdd_mantenimiento_section'
        );
        
        add_settings_field(
            'background_type',
            'Tipo de Fondo',
            array($this, 'background_type_render'),
            'kdd_mantenimiento',
            'kdd_mantenimiento_section'
        );
        
        add_settings_field(
            'background_image',
            'Imagen de Fondo Principal',
            array($this, 'background_image_render'),
            'kdd_mantenimiento',
            'kdd_mantenimiento_section'
        );
        
        add_settings_field(
            'mobile_background_image',
            'Imagen para Móviles',
            array($this, 'mobile_background_image_render'),
            'kdd_mantenimiento',
            'kdd_mantenimiento_section'
        );
        
        add_settings_field(
            'video_type',
            'Plataforma de Video',
            array($this, 'video_type_render'),
            'kdd_mantenimiento',
            'kdd_mantenimiento_section'
        );
        
        add_settings_field(
            'youtube_url',
            'URL de YouTube',
            array($this, 'youtube_url_render'),
            'kdd_mantenimiento',
            'kdd_mantenimiento_section'
        );
        
        add_settings_field(
            'vimeo_url',
            'URL de Vimeo',
            array($this, 'vimeo_url_render'),
            'kdd_mantenimiento',
            'kdd_mantenimiento_section'
        );
        
        add_settings_field(
            'background_color',
            'Color de Fondo',
            array($this, 'background_color_render'),
            'kdd_mantenimiento',
            'kdd_mantenimiento_section'
        );
        
        add_settings_field(
            'logo_url',
            'Logo del Sitio',
            array($this, 'logo_url_render'),
            'kdd_mantenimiento',
            'kdd_mantenimiento_section'
        );
        
        add_settings_field(
            'logo_size',
            'Tamaño del Logo',
            array($this, 'logo_size_render'),
            'kdd_mantenimiento',
            'kdd_mantenimiento_section'
        );
        
        add_settings_field(
            'text_color',
            'Color del Texto',
            array($this, 'text_color_render'),
            'kdd_mantenimiento',
            'kdd_mantenimiento_section'
        );
        
        add_settings_field(
            'custom_message',
            'Mensaje Personalizado',
            array($this, 'custom_message_render'),
            'kdd_mantenimiento',
            'kdd_mantenimiento_section'
        );
        
        // NUEVA SECCIÓN: Redes Sociales (v2.0)
        add_settings_section(
            'kddm_social_section',
            'Redes Sociales',
            array($this, 'social_section_callback'),
            'kdd_mantenimiento'
        );
        
        add_settings_field(
            'social_instagram',
            'Instagram URL',
            array($this, 'social_instagram_render'),
            'kdd_mantenimiento',
            'kddm_social_section'
        );
        
        add_settings_field(
            'social_tiktok',
            'TikTok URL',
            array($this, 'social_tiktok_render'),
            'kdd_mantenimiento',
            'kddm_social_section'
        );
        
        add_settings_field(
            'social_facebook',
            'Facebook URL',
            array($this, 'social_facebook_render'),
            'kdd_mantenimiento',
            'kddm_social_section'
        );
        
        add_settings_field(
            'social_twitter',
            'X (Twitter) URL',
            array($this, 'social_twitter_render'),
            'kdd_mantenimiento',
            'kddm_social_section'
        );
        
        // NUEVA SECCIÓN: WhatsApp (v2.0)
        add_settings_section(
            'kddm_whatsapp_section',
            'WhatsApp',
            array($this, 'whatsapp_section_callback'),
            'kdd_mantenimiento'
        );
        
        add_settings_field(
            'whatsapp_enabled',
            'Activar Botón WhatsApp',
            array($this, 'whatsapp_enabled_render'),
            'kdd_mantenimiento',
            'kddm_whatsapp_section'
        );
        
        add_settings_field(
            'whatsapp_number',
            'Número de WhatsApp',
            array($this, 'whatsapp_number_render'),
            'kdd_mantenimiento',
            'kddm_whatsapp_section'
        );
    }
    
    public function sanitize_settings($input) {
        $sanitized = array();
        
        // Campos existentes
        $sanitized['enable_maintenance'] = isset($input['enable_maintenance']) ? '1' : '0';
        $sanitized['background_type'] = sanitize_text_field($input['background_type']);
        $sanitized['background_image'] = esc_url_raw($input['background_image']);
        $sanitized['mobile_background_image'] = esc_url_raw($input['mobile_background_image']);
        $sanitized['video_type'] = sanitize_text_field($input['video_type']);
        $sanitized['youtube_url'] = esc_url_raw($input['youtube_url']);
        $sanitized['vimeo_url'] = esc_url_raw($input['vimeo_url']);
        $sanitized['background_color'] = sanitize_hex_color($input['background_color']);
        $sanitized['logo_url'] = esc_url_raw($input['logo_url']);
        $sanitized['logo_size'] = sanitize_text_field($input['logo_size']);
        $sanitized['text_color'] = sanitize_hex_color($input['text_color']);
        $sanitized['mobile_fallback'] = isset($input['mobile_fallback']) ? '1' : '0';
        
        // NUEVO: Editor de texto enriquecido (v2.0)
        $sanitized['custom_message'] = wp_kses_post($input['custom_message']);
        
        // NUEVO: Redes sociales (v2.0)
        $sanitized['social_instagram'] = esc_url_raw($input['social_instagram']);
        $sanitized['social_tiktok'] = esc_url_raw($input['social_tiktok']);
        $sanitized['social_facebook'] = esc_url_raw($input['social_facebook']);
        $sanitized['social_twitter'] = esc_url_raw($input['social_twitter']);
        
        // NUEVO: WhatsApp (v2.0)
        $sanitized['whatsapp_enabled'] = isset($input['whatsapp_enabled']) ? '1' : '0';
        $sanitized['whatsapp_number'] = sanitize_text_field($input['whatsapp_number']);
        
        return $sanitized;
    }
    
    // Función para extraer ID de YouTube
    private function get_youtube_id($url) {
        $pattern = '/(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/)([^"&?\/\s]{11})/';
        preg_match($pattern, $url, $matches);
        return isset($matches[1]) ? $matches[1] : false;
    }
    
    // Función para extraer ID de Vimeo
    private function get_vimeo_id($url) {
        $pattern = '/vimeo\.com\/(?:channels\/(?:\w+\/)?|groups\/(?:[^\/]*)\/videos\/|)(\d+)(?:|\/\?)/';
        preg_match($pattern, $url, $matches);
        return isset($matches[1]) ? $matches[1] : false;
    }
    
    // Función para detectar dispositivo móvil
    private function is_mobile_device() {
        return wp_is_mobile();
    }
    
    // Render functions for each field - CAMPOS EXISTENTES
    public function enable_maintenance_render() {
        $enabled = isset($this->options['enable_maintenance']) && $this->options['enable_maintenance'] == '1';
        ?>
        <label>
            <input type='checkbox' name='kdd_mantenimiento_settings[enable_maintenance]' <?php checked($enabled); ?> value='1'>
            <strong>Activar el modo mantenimiento</strong>
        </label>
        <p class="description">Cuando esté activado, los visitantes verán la página de mantenimiento. Los administradores seguirán viendo el sitio normal.</p>
        <?php
    }
    
    public function background_type_render() {
        $type = isset($this->options['background_type']) ? $this->options['background_type'] : 'image';
        ?>
        <select name='kdd_mantenimiento_settings[background_type]' id='kdd_background_type'>
            <option value='image' <?php selected($type, 'image'); ?>>Imagen de Fondo</option>
            <option value='video' <?php selected($type, 'video'); ?>>Video de Fondo (YouTube/Vimeo)</option>
            <option value='color' <?php selected($type, 'color'); ?>>Color Sólido</option>
        </select>
        <p class="description">Selecciona el tipo de fondo para la página de mantenimiento.</p>
        
        <?php if ($type === 'video'): ?>
        <div class="kdd-notice kdd-warning">
            <p><strong>💡 Nota importante sobre videos en móviles:</strong> Los videos de fondo pueden tener problemas de reproducción en dispositivos móviles. 
            Por esta razón, <strong>en móviles se mostrará automáticamente la imagen de fondo para móviles</strong> que configures a continuación.</p>
        </div>
        <?php endif; ?>
        <?php
    }
    
    public function background_image_render() {
        $image_url = isset($this->options['background_image']) ? $this->options['background_image'] : '';
        ?>
        <div class="kdd-media-upload">
            <input type='text' id='kdd_background_image' name='kdd_mantenimiento_settings[background_image]' value='<?php echo esc_url($image_url); ?>' style='width: 70%;' placeholder='https://...'>
            <button type='button' class='button kdd-media-btn' data-target='kdd_background_image' data-type='image'>Seleccionar Imagen</button>
            <button type='button' class='button kdd-remove-btn' data-target='kdd_background_image'>Quitar</button>
            
            <?php if ($image_url): ?>
                <div class="kdd-preview">
                    <img src="<?php echo esc_url($image_url); ?>" style="max-width: 300px; height: auto; margin-top: 10px; border: 2px solid #ddd; border-radius: 5px;">
                    <p class="description">Vista previa de la imagen de fondo principal</p>
                </div>
            <?php else: ?>
                <div class="kdd-preview" style="display: none;">
                    <img src="" style="max-width: 300px; height: auto; margin-top: 10px; border: 2px solid #ddd; border-radius: 5px;">
                    <p class="description">Vista previa de la imagen de fondo principal</p>
                </div>
            <?php endif; ?>
        </div>
        <p class="description">Selecciona una imagen para el fondo principal. Recomendado: 1920x1080px o similar.</p>
        <?php
    }
    
    public function mobile_background_image_render() {
        $mobile_image_url = isset($this->options['mobile_background_image']) ? $this->options['mobile_background_image'] : '';
        $background_type = isset($this->options['background_type']) ? $this->options['background_type'] : 'image';
        ?>
        <div class="kdd-media-upload">
            <input type='text' id='kdd_mobile_background_image' name='kdd_mantenimiento_settings[mobile_background_image]' value='<?php echo esc_url($mobile_image_url); ?>' style='width: 70%;' placeholder='https://...'>
            <button type='button' class='button kdd-media-btn' data-target='kdd_mobile_background_image' data-type='image'>Seleccionar Imagen</button>
            <button type='button' class='button kdd-remove-btn' data-target='kdd_mobile_background_image'>Quitar</button>
            
            <?php if ($mobile_image_url): ?>
                <div class="kdd-preview">
                    <img src="<?php echo esc_url($mobile_image_url); ?>" style="max-width: 300px; height: auto; margin-top: 10px; border: 2px solid #ddd; border-radius: 5px;">
                    <p class="description">Vista previa de la imagen para móviles</p>
                </div>
            <?php else: ?>
                <div class="kdd-preview" style="display: none;">
                    <img src="" style="max-width: 300px; height: auto; margin-top: 10px; border: 2px solid #ddd; border-radius: 5px;">
                    <p class="description">Vista previa de la imagen para móviles</p>
                </div>
            <?php endif; ?>
        </div>
        
        <?php if ($background_type === 'video'): ?>
            <p class="description"><strong>Esta imagen se usará en dispositivos móviles</strong> cuando el fondo sea un video. En escritorio se mostrará el video normal.</p>
        <?php else: ?>
            <p class="description">Imagen opcional para dispositivos móviles. Si no se selecciona, se usará la imagen principal.</p>
        <?php endif; ?>
        <?php
    }
    
    public function video_type_render() {
        $video_type = isset($this->options['video_type']) ? $this->options['video_type'] : 'youtube';
        ?>
        <select name='kdd_mantenimiento_settings[video_type]' id='kdd_video_type'>
            <option value='youtube' <?php selected($video_type, 'youtube'); ?>>YouTube</option>
            <option value='vimeo' <?php selected($video_type, 'vimeo'); ?>>Vimeo</option>
        </select>
        <p class="description">Selecciona la plataforma del video de fondo.</p>
        <?php
    }
    
    public function youtube_url_render() {
        $youtube_url = isset($this->options['youtube_url']) ? $this->options['youtube_url'] : '';
        $youtube_id = $this->get_youtube_id($youtube_url);
        ?>
        <div class="kdd-video-upload">
            <input type='text' id='kdd_youtube_url' name='kdd_mantenimiento_settings[youtube_url]' value='<?php echo esc_url($youtube_url); ?>' style='width: 70%;' placeholder='https://www.youtube.com/watch?v=... o https://youtu.be/...'>
            <button type='button' class='button kdd-test-video' data-type='youtube'>Probar Video</button>
            
            <?php if ($youtube_id): ?>
                <div class="kdd-preview">
                    <div class="kdd-video-preview">
                        <img src="https://img.youtube.com/vi/<?php echo esc_attr($youtube_id); ?>/maxresdefault.jpg" style="max-width: 300px; height: auto; margin-top: 10px; border: 2px solid #ddd; border-radius: 5px;" alt="Vista previa de YouTube">
                        <p class="description">Vista previa del video de YouTube</p>
                        <p class="description"><small>ID del video: <?php echo esc_html($youtube_id); ?></small></p>
                    </div>
                </div>
            <?php else: ?>
                <div class="kdd-preview" style="display: none;">
                    <div class="kdd-video-preview">
                        <img src="" style="max-width: 300px; height: auto; margin-top: 10px; border: 2px solid #ddd; border-radius: 5px;" alt="Vista previa de YouTube">
                        <p class="description">Vista previa del video de YouTube</p>
                    </div>
                </div>
            <?php endif; ?>
        </div>
        <p class="description">Pega la URL completa del video de YouTube. Formatos aceptados: youtube.com/watch?v=ID o youtu.be/ID</p>
        <?php
    }
    
    public function vimeo_url_render() {
        $vimeo_url = isset($this->options['vimeo_url']) ? $this->options['vimeo_url'] : '';
        $vimeo_id = $this->get_vimeo_id($vimeo_url);
        ?>
        <div class="kdd-video-upload">
            <input type='text' id='kdd_vimeo_url' name='kdd_mantenimiento_settings[vimeo_url]' value='<?php echo esc_url($vimeo_url); ?>' style='width: 70%;' placeholder='https://vimeo.com/...'>
            <button type='button' class='button kdd-test-video' data-type='vimeo'>Probar Video</button>
            
            <?php if ($vimeo_id): ?>
                <div class="kdd-preview">
                    <div class="kdd-video-preview">
                        <?php
                        // Intentar obtener thumbnail de Vimeo
                        $vimeo_thumb = false;
                        if ($vimeo_id) {
                            $vimeo_data = wp_remote_get("https://vimeo.com/api/v2/video/{$vimeo_id}.json");
                            if (!is_wp_error($vimeo_data) && wp_remote_retrieve_response_code($vimeo_data) === 200) {
                                $vimeo_info = json_decode(wp_remote_retrieve_body($vimeo_data), true);
                                if (!empty($vimeo_info[0]['thumbnail_large'])) {
                                    $vimeo_thumb = $vimeo_info[0]['thumbnail_large'];
                                }
                            }
                        }
                        ?>
                        <?php if ($vimeo_thumb): ?>
                            <img src="<?php echo esc_url($vimeo_thumb); ?>" style="max-width: 300px; height: auto; margin-top: 10px; border: 2px solid #ddd; border-radius: 5px;" alt="Vista previa de Vimeo">
                        <?php else: ?>
                            <div style="max-width: 300px; height: 150px; margin-top: 10px; border: 2px solid #ddd; border-radius: 5px; background: #f0f0f0; display: flex; align-items: center; justify-content: center;">
                                <span>Vista previa no disponible</span>
                            </div>
                        <?php endif; ?>
                        <p class="description">Vista previa del video de Vimeo</p>
                        <p class="description"><small>ID del video: <?php echo esc_html($vimeo_id); ?></small></p>
                    </div>
                </div>
            <?php else: ?>
                <div class="kdd-preview" style="display: none;">
                    <div class="kdd-video-preview">
                        <div style="max-width: 300px; height: 150px; margin-top: 10px; border: 2px solid #ddd; border-radius: 5px; background: #f0f0f0; display: flex; align-items: center; justify-content: center;">
                            <span>Vista previa no disponible</span>
                        </div>
                        <p class="description">Vista previa del video de Vimeo</p>
                    </div>
                </div>
            <?php endif; ?>
        </div>
        <p class="description">Pega la URL completa del video de Vimeo. Formato: vimeo.com/ID</p>
        <?php
    }
    
    public function background_color_render() {
        $color = isset($this->options['background_color']) ? $this->options['background_color'] : '#2c3e50';
        ?>
        <input type='text' name='kdd_mantenimiento_settings[background_color]' value='<?php echo esc_attr($color); ?>' class='kdd-color-field'>
        <p class="description">Selecciona un color sólido para el fondo.</p>
        <?php
    }
    
    public function logo_url_render() {
        $logo_url = isset($this->options['logo_url']) ? $this->options['logo_url'] : '';
        ?>
        <div class="kdd-media-upload">
            <input type='text' id='kdd_logo_url' name='kdd_mantenimiento_settings[logo_url]' value='<?php echo esc_url($logo_url); ?>' style='width: 70%;' placeholder='https://...'>
            <button type='button' class='button kdd-media-btn' data-target='kdd_logo_url' data-type='image'>Seleccionar Logo</button>
            <button type='button' class='button kdd-remove-btn' data-target='kdd_logo_url'>Quitar</button>
            
            <?php if ($logo_url): ?>
                <div class="kdd-preview">
                    <img src="<?php echo esc_url($logo_url); ?>" style="max-width: 300px; height: auto; margin-top: 10px; border: 2px solid #ddd; border-radius: 5px; background: white; padding: 10px;">
                    <p class="description">Vista previa del logo</p>
                </div>
            <?php else: ?>
                <div class="kdd-preview" style="display: none;">
                    <img src="" style="max-width: 300px; height: auto; margin-top: 10px; border: 2px solid #ddd; border-radius: 5px; background: white; padding: 10px;">
                    <p class="description">Vista previa del logo</p>
                </div>
            <?php endif; ?>
        </div>
        <p class="description">Selecciona o sube el logo que se mostrará en la página de mantenimiento. Formatos recomendados: PNG, JPG, SVG.</p>
        <?php
    }
    
    public function logo_size_render() {
        $logo_size = isset($this->options['logo_size']) ? $this->options['logo_size'] : '200px';
        $sizes = array(
            '150px' => 'Pequeño (150px)',
            '200px' => 'Mediano (200px)',
            '250px' => 'Grande (250px)',
            '300px' => 'Extra Grande (300px)',
            '350px' => 'Jumbo (350px)'
        );
        ?>
        <select name='kdd_mantenimiento_settings[logo_size]'>
            <?php foreach ($sizes as $value => $label): ?>
                <option value='<?php echo esc_attr($value); ?>' <?php selected($logo_size, $value); ?>><?php echo esc_html($label); ?></option>
            <?php endforeach; ?>
        </select>
        <p class="description">Selecciona el tamaño que tendrá el logo en la página de mantenimiento.</p>
        <?php
    }
    
    public function text_color_render() {
        $color = isset($this->options['text_color']) ? $this->options['text_color'] : '#ffffff';
        ?>
        <input type='text' name='kdd_mantenimiento_settings[text_color]' value='<?php echo esc_attr($color); ?>' class='kdd-color-field'>
        <p class="description">Color del texto y elementos principales.</p>
        <?php
    }
    
    // NUEVO: Editor de texto enriquecido (v2.0)
    public function custom_message_render() {
        $message = isset($this->options['custom_message']) ? $this->options['custom_message'] : 'Estamos realizando tareas de mantenimiento para mejorar tu experiencia. Volveremos pronto.';
        ?>
        <div class="kdd-rich-text-editor">
            <?php
            $editor_id = 'kdd_custom_message';
            $settings = array(
                'textarea_name' => 'kdd_mantenimiento_settings[custom_message]',
                'textarea_rows' => 8,
                'media_buttons' => false,
                'tinymce' => array(
                    'toolbar1' => 'bold,italic,underline,link,unlink,bullist,numlist,alignleft,aligncenter,alignright',
                    'toolbar2' => '',
                    'content_css' => plugins_url('assets/css/editor.css', __FILE__)
                ),
                'quicktags' => false,
                'editor_height' => 200,
            );
            wp_editor(wp_kses_post($message), $editor_id, $settings);
            ?>
            <p class="description">Este mensaje se mostrará a los visitantes mientras el sitio está en mantenimiento. Puedes usar formato de texto enriquecido.</p>
        </div>
        <?php
    }
    
    // NUEVAS SECCIONES: Redes Sociales (v2.0)
    public function social_section_callback() {
        echo '<div class="kdd-section-description">';
        echo '<p>Configura los enlaces a tus redes sociales. Los iconos se mostrarán <strong>solo si tienen URL configurada</strong>, en diseño horizontal centrado sobre el pie de página.</p>';
        echo '</div>';
    }
    
    public function social_instagram_render() {
        $url = isset($this->options['social_instagram']) ? $this->options['social_instagram'] : '';
        ?>
        <input type='text' name='kdd_mantenimiento_settings[social_instagram]' value='<?php echo esc_url($url); ?>' style='width: 70%;' placeholder='https://instagram.com/tu-usuario'>
        <p class="description">URL completa de tu perfil de Instagram</p>
        <?php
    }
    
    public function social_tiktok_render() {
        $url = isset($this->options['social_tiktok']) ? $this->options['social_tiktok'] : '';
        ?>
        <input type='text' name='kdd_mantenimiento_settings[social_tiktok]' value='<?php echo esc_url($url); ?>' style='width: 70%;' placeholder='https://tiktok.com/@tu-usuario'>
        <p class="description">URL completa de tu perfil de TikTok</p>
        <?php
    }
    
    public function social_facebook_render() {
        $url = isset($this->options['social_facebook']) ? $this->options['social_facebook'] : '';
        ?>
        <input type='text' name='kdd_mantenimiento_settings[social_facebook]' value='<?php echo esc_url($url); ?>' style='width: 70%;' placeholder='https://facebook.com/tu-pagina'>
        <p class="description">URL completa de tu página de Facebook</p>
        <?php
    }
    
    public function social_twitter_render() {
        $url = isset($this->options['social_twitter']) ? $this->options['social_twitter'] : '';
        ?>
        <input type='text' name='kdd_mantenimiento_settings[social_twitter]' value='<?php echo esc_url($url); ?>' style='width: 70%;' placeholder='https://x.com/tu-usuario'>
        <p class="description">URL completa de tu perfil de X (Twitter)</p>
        <?php
    }
    
    // NUEVAS SECCIONES: WhatsApp (v2.0)
    public function whatsapp_section_callback() {
        echo '<div class="kdd-section-description">';
        echo '<p>Configura el botón flotante de WhatsApp que aparecerá en la <strong>esquina inferior derecha</strong> de la página de mantenimiento.</p>';
        echo '</div>';
    }
    
    public function whatsapp_enabled_render() {
        $enabled = isset($this->options['whatsapp_enabled']) && $this->options['whatsapp_enabled'] == '1';
        ?>
        <label>
            <input type='checkbox' name='kdd_mantenimiento_settings[whatsapp_enabled]' <?php checked($enabled); ?> value='1'>
            <strong>Activar botón flotante de WhatsApp</strong>
        </label>
        <p class="description">Cuando esté activado, se mostrará un botón flotante de WhatsApp en la página de mantenimiento.</p>
        <?php
    }
    
    public function whatsapp_number_render() {
        $number = isset($this->options['whatsapp_number']) ? $this->options['whatsapp_number'] : '';
        ?>
        <input type='text' name='kdd_mantenimiento_settings[whatsapp_number]' value='<?php echo esc_attr($number); ?>' style='width: 70%;' placeholder='+573001234567'>
        <p class="description">Número de WhatsApp con código de país (ej: +573001234567 para Colombia)</p>
        <?php
    }
    
    public function settings_section_callback() {
        echo '<div class="kdd-section-description">';
        echo '<p>Configura los ajustes para la página de mantenimiento de tu sitio web. <strong>Los administradores siempre podrán ver el sitio normal.</strong></p>';
        echo '</div>';
    }
    
    public function options_page() {
        ?>
        <div class="wrap kdd-settings-wrap">
            <h1><span class="dashicons dashicons-admin-tools"></span> KDD Mantenimiento - Configuración v2.0</h1>
            
            <?php if (isset($_GET['settings-updated'])): ?>
                <div class="notice notice-success is-dismissible">
                    <p>Configuración de KDD Mantenimiento guardada correctamente.</p>
                </div>
            <?php endif; ?>
            
            <div class="kdd-settings-container">
                <div class="kdd-settings-main">
                    <form action='options.php' method='post'>
                        <?php
                        settings_fields('kdd_mantenimiento');
                        do_settings_sections('kdd_mantenimiento');
                        submit_button('Guardar Configuración', 'primary', 'submit', false);
                        ?>
                    </form>
                </div>
                
                <div class="kdd-settings-sidebar">
                    <div class="kdd-info-box">
                        <h3><span class="dashicons dashicons-info"></span> Información del Plugin</h3>
                        <p><strong>Nombre:</strong> KDDMantenimiento</p>
                        <p><strong>Versión:</strong> 2.0</p>
                        <p><strong>Autor:</strong> KDD Business</p>
                        <p><strong>Para:</strong> KDDClub</p>
                    </div>
                    
                    <div class="kdd-info-box">
                        <h3><span class="dashicons dashicons-visibility"></span> Nuevo en v2.0</h3>
                        <ul>
                            <li>✅ Editor de texto enriquecido</li>
                            <li>✅ Redes sociales (Instagram, TikTok, Facebook, X)</li>
                            <li>✅ Botón flotante de WhatsApp</li>
                            <li>✅ Diseño horizontal para redes sociales</li>
                            <li>✅ Tooltip "Chatear en WhatsApp"</li>
                        </ul>
                    </div>
                    
                    <div class="kdd-info-box">
                        <h3><span class="dashicons dashicons-format-video"></span> Videos Soporte</h3>
                        <p><strong>YouTube:</strong> Soporta enlaces completos y youtu.be</p>
                        <p><strong>Vimeo:</strong> Soporta enlaces estándar de Vimeo</p>
                        <p><strong>Móviles:</strong> Se muestra imagen en lugar de video</p>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    private function show_maintenance_page() {
        // Configuración
        $background_type = isset($this->options['background_type']) ? $this->options['background_type'] : 'image';
        $background_image = isset($this->options['background_image']) ? $this->options['background_image'] : '';
        $mobile_background_image = isset($this->options['mobile_background_image']) ? $this->options['mobile_background_image'] : '';
        $video_type = isset($this->options['video_type']) ? $this->options['video_type'] : 'youtube';
        $youtube_url = isset($this->options['youtube_url']) ? $this->options['youtube_url'] : '';
        $vimeo_url = isset($this->options['vimeo_url']) ? $this->options['vimeo_url'] : '';
        $background_color = isset($this->options['background_color']) ? $this->options['background_color'] : '#2c3e50';
        $logo_url = isset($this->options['logo_url']) ? $this->options['logo_url'] : '';
        $logo_size = isset($this->options['logo_size']) ? $this->options['logo_size'] : '200px';
        $custom_message = isset($this->options['custom_message']) ? $this->options['custom_message'] : 'Estamos realizando tareas de mantenimiento para mejorar tu experiencia. Volveremos pronto.';
        $text_color = isset($this->options['text_color']) ? $this->options['text_color'] : '#ffffff';
        
        // NUEVO: Redes sociales (v2.0)
        $social_instagram = isset($this->options['social_instagram']) ? $this->options['social_instagram'] : '';
        $social_tiktok = isset($this->options['social_tiktok']) ? $this->options['social_tiktok'] : '';
        $social_facebook = isset($this->options['social_facebook']) ? $this->options['social_facebook'] : '';
        $social_twitter = isset($this->options['social_twitter']) ? $this->options['social_twitter'] : '';
        
        // NUEVO: WhatsApp (v2.0)
        $whatsapp_enabled = isset($this->options['whatsapp_enabled']) && $this->options['whatsapp_enabled'] == '1';
        $whatsapp_number = isset($this->options['whatsapp_number']) ? $this->options['whatsapp_number'] : '';
        
        // Extraer IDs de video
        $youtube_id = $this->get_youtube_id($youtube_url);
        $vimeo_id = $this->get_vimeo_id($vimeo_url);
        
        // Determinar si es dispositivo móvil
        $is_mobile = $this->is_mobile_device();
        
        // Headers HTTP
        status_header(503);
        nocache_headers();
        
        // HTML de la página de mantenimiento
        ?>
        <!DOCTYPE html>
        <html <?php language_attributes(); ?>>
        <head>
            <meta charset="<?php bloginfo('charset'); ?>">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title><?php echo esc_html(get_bloginfo('name')); ?> - Sitio en Mantenimiento</title>
            
            <?php if ($background_type === 'video' && !$is_mobile): ?>
            <style>.kdd-video-background { pointer-events: none; }</style>
            <?php endif; ?>
            
            <style>
                * {
                    margin: 0;
                    padding: 0;
                    box-sizing: border-box;
                }
                
                body {
                    font-family: 'Arial', 'Helvetica', sans-serif;
                    min-height: 100vh;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    position: relative;
                    overflow: hidden;
                    color: <?php echo esc_attr($text_color); ?>;
                    line-height: 1.6;
                }
                
                .kdd-background-overlay {
                    position: absolute;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    background: rgba(0, 0, 0, 0.6);
                    z-index: -1;
                }
                
                .kdd-background-image {
                    position: absolute;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    object-fit: cover;
                    z-index: -2;
                }
                
                .kdd-video-background {
                    position: absolute;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    z-index: -2;
                    overflow: hidden;
                }
                
                .kdd-video-background iframe {
                    position: absolute;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    border: none;
                    /* Solución para cubrir completamente sin franjas */
                    object-fit: cover;
                    /* Asegurar que cubra todo el espacio */
                    min-width: 100%;
                    min-height: 100%;
                    transform: scale(1.09); /* Aplicado tanto para YouTube como Vimeo */
                }
                
                .kdd-maintenance-container {
                    text-align: center;
                    padding: 50px 40px;
                    max-width: 800px;
                    background: rgba(255, 255, 255, 0.1);
                    backdrop-filter: blur(15px);
                    border-radius: 20px;
                    margin: 20px;
                    border: 1px solid rgba(255, 255, 255, 0.2);
                    box-shadow: 0 8px 32px rgba(0, 0, 0, 0.3);
                }
                
                .kdd-logo {
                    margin-bottom: 40px;
                }
                
                .kdd-logo img {
                    max-width: 100%;
                    height: auto;
                    border-radius: 12px;
                    /* Sombra eliminada como solicitado */
                }
                
                .kdd-message {
                    font-size: 1.4em;
                    line-height: 1.7;
                    margin-bottom: 40px;
                    text-shadow: 1px 1px 3px rgba(0, 0, 0, 0.5);
                    font-weight: 300;
                }
                
                .kdd-message p {
                    margin-bottom: 1em;
                }
                
                .kdd-message p:last-child {
                    margin-bottom: 0;
                }
                
                .kdd-loading-bar {
                    width: 100%;
                    height: 6px;
                    background: rgba(255, 255, 255, 0.3);
                    border-radius: 3px;
                    overflow: hidden;
                    margin: 40px 0;
                }
                
                .kdd-loading-progress {
                    width: 100%;
                    height: 100%;
                    background: <?php echo esc_attr($text_color); ?>;
                    animation: kdd-loading 2.5s infinite ease-in-out;
                    transform-origin: left;
                }
                
                .kdd-footer {
                    margin-top: 30px;
                    font-size: 0.9em;
                    opacity: 0.8;
                    padding-top: 20px;
                    border-top: 1px solid rgba(255, 255, 255, 0.2);
                }
                
                .kdd-site-name {
                    font-weight: 600;
                    font-size: 1.1em;
                }
                
                /* NUEVO: Estilos para redes sociales (v2.0) */
                .kdd-social-links {
                    display: flex;
                    justify-content: center;
                    gap: 20px;
                    margin-bottom: 20px;
                }
                
                .kdd-social-icon {
                    display: inline-block;
                    width: 40px;
                    height: 40px;
                    border-radius: 50%;
                    background: rgba(255, 255, 255, 0.2);
                    backdrop-filter: blur(10px);
                    border: 1px solid rgba(255, 255, 255, 0.3);
                    transition: all 0.3s ease;
                    position: relative;
                }
                
                .kdd-social-icon:hover {
                    background: rgba(255, 255, 255, 0.3);
                    transform: translateY(-2px);
                }
                
                .kdd-social-icon svg {
                    width: 20px;
                    height: 20px;
                    position: absolute;
                    top: 50%;
                    left: 50%;
                    transform: translate(-50%, -50%);
                    fill: <?php echo esc_attr($text_color); ?>;
                }
                
                /* NUEVO: Estilos para WhatsApp flotante (v2.0) */
                .kdd-whatsapp-float {
                    position: fixed;
                    bottom: 25px;
                    right: 25px;
                    width: 60px;
                    height: 60px;
                    background: #25D366;
                    border-radius: 50%;
                    box-shadow: 0 4px 20px rgba(37, 211, 102, 0.4);
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    z-index: 1000;
                    transition: all 0.3s ease;
                    animation: kdd-pulse 2s infinite;
                }
                
                .kdd-whatsapp-float:hover {
                    transform: scale(1.1);
                    box-shadow: 0 6px 25px rgba(37, 211, 102, 0.6);
                    animation: none;
                }
                
                .kdd-whatsapp-float svg {
                    width: 30px;
                    height: 30px;
                    fill: white;
                }
                
                .kdd-whatsapp-tooltip {
                    position: absolute;
                    bottom: 70px;
                    right: 0;
                    background: rgba(0, 0, 0, 0.8);
                    color: white;
                    padding: 8px 12px;
                    border-radius: 6px;
                    font-size: 12px;
                    white-space: nowrap;
                    opacity: 0;
                    transform: translateY(10px);
                    transition: all 0.3s ease;
                    pointer-events: none;
                }
                
                .kdd-whatsapp-float:hover .kdd-whatsapp-tooltip {
                    opacity: 1;
                    transform: translateY(0);
                }
                
                @keyframes kdd-loading {
                    0% { 
                        transform: scaleX(0);
                        opacity: 0.7;
                    }
                    50% {
                        transform: scaleX(1);
                        opacity: 1;
                    }
                    100% { 
                        transform: scaleX(0);
                        opacity: 0.7;
                    }
                }
                
                @keyframes kdd-pulse {
                    0% {
                        box-shadow: 0 4px 20px rgba(37, 211, 102, 0.4);
                    }
                    50% {
                        box-shadow: 0 4px 25px rgba(37, 211, 102, 0.8);
                    }
                    100% {
                        box-shadow: 0 4px 20px rgba(37, 211, 102, 0.4);
                    }
                }
                
                /* Solución específica para Vimeo - eliminar bordes */
                .kdd-video-background iframe[src*="vimeo"] {
                    width: 100% !important;
                    height: 100% !important;
                }
                
                /* Ajustes para móviles */
                @media (max-width: 768px) {
                    .kdd-maintenance-container {
                        padding: 30px 20px;
                        margin: 15px;
                        border-radius: 15px;
                    }
                    
                    .kdd-message {
                        font-size: 1.2em;
                    }
                    
                    .kdd-logo {
                        margin-bottom: 30px;
                    }
                    
                    /* Ocultar videos en móviles */
                    .kdd-video-background {
                        display: none !important;
                    }
                    
                    /* Mostrar imagen de fondo en móviles */
                    .kdd-mobile-background {
                        display: block !important;
                    }
                    
                    /* Ajustes WhatsApp móvil */
                    .kdd-whatsapp-float {
                        bottom: 20px;
                        right: 20px;
                        width: 55px;
                        height: 55px;
                    }
                    
                    .kdd-whatsapp-float svg {
                        width: 28px;
                        height: 28px;
                    }
                    
                    .kdd-social-links {
                        gap: 15px;
                    }
                    
                    .kdd-social-icon {
                        width: 35px;
                        height: 35px;
                    }
                    
                    .kdd-social-icon svg {
                        width: 18px;
                        height: 18px;
                    }
                }
                
                @media (max-width: 480px) {
                    .kdd-maintenance-container {
                        padding: 25px 15px;
                    }
                    
                    .kdd-message {
                        font-size: 1.1em;
                    }
                    
                    .kdd-social-links {
                        gap: 12px;
                    }
                    
                    .kdd-social-icon {
                        width: 32px;
                        height: 32px;
                    }
                    
                    .kdd-social-icon svg {
                        width: 16px;
                        height: 16px;
                    }
                }
                
                /* Clase para imagen de fondo móvil */
                .kdd-mobile-background {
                    display: none;
                }
            </style>
        </head>
        <body>
            <?php 
            // Lógica para mostrar el fondo según el dispositivo
            if ($is_mobile && $mobile_background_image && $background_type === 'video'): 
                // Móvil con imagen específica para móviles y fondo tipo video
                ?>
                <img src="<?php echo esc_url($mobile_background_image); ?>" class="kdd-background-image kdd-mobile-background" alt="Background">
            <?php elseif ($is_mobile && $background_image && $background_type === 'video'): 
                // Móvil con imagen principal y fondo tipo video
                ?>
                <img src="<?php echo esc_url($background_image); ?>" class="kdd-background-image kdd-mobile-background" alt="Background">
            <?php elseif ($background_type === 'image' && $background_image): 
                // Escritorio o móvil con imagen principal
                ?>
                <img src="<?php echo esc_url($background_image); ?>" class="kdd-background-image" alt="Background">
            <?php elseif ($background_type === 'video' && $video_type === 'youtube' && $youtube_id && !$is_mobile): 
                // Escritorio con YouTube
                ?>
                <div class="kdd-video-background">
                    <iframe 
                        src="https://www.youtube.com/embed/<?php echo esc_attr($youtube_id); ?>?autoplay=1&mute=1&loop=1&playlist=<?php echo esc_attr($youtube_id); ?>&controls=0&showinfo=0&rel=0&modestbranding=1&playsinline=1&enablejsapi=1&origin=<?php echo esc_url(home_url()); ?>" 
                        frameborder="0" 
                        allow="autoplay; encrypted-media" 
                        allowfullscreen
                    ></iframe>
                </div>
            <?php elseif ($background_type === 'video' && $video_type === 'vimeo' && $vimeo_id && !$is_mobile): 
                // Escritorio con Vimeo
                ?>
                <div class="kdd-video-background">
                    <iframe 
                        src="https://player.vimeo.com/video/<?php echo esc_attr($vimeo_id); ?>?background=1&autoplay=1&loop=1&muted=1&autopause=0&transparent=1"
                        frameborder="0" 
                        allow="autoplay; fullscreen" 
                        allowfullscreen
                    ></iframe>
                </div>
            <?php else: 
                // Color de fondo por defecto
                ?>
                <div style="position: absolute; top: 0; left: 0; width: 100%; height: 100%; background: <?php echo esc_attr($background_color); ?>; z-index: -2;"></div>
            <?php endif; ?>
            
            <div class="kdd-background-overlay"></div>
            
            <div class="kdd-maintenance-container">
                <?php if ($logo_url): ?>
                    <div class="kdd-logo">
                        <img src="<?php echo esc_url($logo_url); ?>" alt="<?php echo esc_attr(get_bloginfo('name')); ?> Logo" style="width: <?php echo esc_attr($logo_size); ?>; height: auto;">
                    </div>
                <?php endif; ?>
                
                <div class="kdd-message">
                    <?php echo wp_kses_post(wpautop($custom_message)); ?>
                </div>
                
                <div class="kdd-loading-bar">
                    <div class="kdd-loading-progress"></div>
                </div>
                
                <?php
                // NUEVO: Mostrar redes sociales si hay al menos una configurada (v2.0)
                $has_social_links = $social_instagram || $social_tiktok || $social_facebook || $social_twitter;
                if ($has_social_links):
                ?>
                    <div class="kdd-social-links">
                        <?php if ($social_instagram): ?>
                            <a href="<?php echo esc_url($social_instagram); ?>" target="_blank" class="kdd-social-icon" title="Instagram">
                                <svg viewBox="0 0 24 24">
                                    <path d="M12 2.163c3.204 0 3.584.012 4.85.07 3.252.148 4.771 1.691 4.919 4.919.058 1.265.069 1.645.069 4.849 0 3.205-.012 3.584-.069 4.849-.149 3.225-1.664 4.771-4.919 4.919-1.266.058-1.644.07-4.85.07-3.204 0-3.584-.012-4.849-.07-3.26-.149-4.771-1.699-4.919-4.92-.058-1.265-.07-1.644-.07-4.849 0-3.204.013-3.583.07-4.849.149-3.227 1.664-4.771 4.919-4.919 1.266-.057 1.645-.069 4.849-.069zm0-2.163c-3.259 0-3.667.014-4.947.072-4.358.2-6.78 2.618-6.98 6.98-.059 1.281-.073 1.689-.073 4.948 0 3.259.014 3.668.072 4.948.2 4.358 2.618 6.78 6.98 6.98 1.281.058 1.689.072 4.948.072 3.259 0 3.668-.014 4.948-.072 4.354-.2 6.782-2.618 6.979-6.98.059-1.28.073-1.689.073-4.948 0-3.259-.014-3.667-.072-4.947-.196-4.354-2.617-6.78-6.979-6.98-1.281-.059-1.69-.073-4.949-.073zm0 5.838c-3.403 0-6.162 2.759-6.162 6.162s2.759 6.163 6.162 6.163 6.162-2.759 6.162-6.163c0-3.403-2.759-6.162-6.162-6.162zm0 10.162c-2.209 0-4-1.79-4-4 0-2.209 1.791-4 4-4s4 1.791 4 4c0 2.21-1.791 4-4 4zm6.406-11.845c-.796 0-1.441.645-1.441 1.44s.645 1.44 1.441 1.44c.795 0 1.439-.645 1.439-1.44s-.644-1.44-1.439-1.44z"/>
                                </svg>
                            </a>
                        <?php endif; ?>
                        
                        <?php if ($social_tiktok): ?>
                            <a href="<?php echo esc_url($social_tiktok); ?>" target="_blank" class="kdd-social-icon" title="TikTok">
                                <svg viewBox="0 0 24 24">
                                    <path d="M12.525.02c1.31-.02 2.61-.01 3.91-.02.08 1.53.63 3.09 1.75 4.17 1.12 1.11 2.7 1.62 4.24 1.79v4.03c-1.44-.05-2.89-.35-4.2-.97-.57-.26-1.1-.59-1.62-.93-.01 2.92.01 5.84-.02 8.75-.08 1.4-.54 2.79-1.35 3.94-1.31 1.92-3.58 3.17-5.91 3.21-1.43.08-2.86-.31-4.08-1.03-2.02-1.19-3.44-3.37-3.65-5.71-.02-.5-.03-1-.01-1.49.18-1.9 1.12-3.72 2.58-4.96 1.66-1.44 3.98-2.13 6.15-1.72.02 1.48-.04 2.96-.04 4.44-.99-.32-2.15-.23-3.02.37-.63.41-1.11 1.04-1.36 1.75-.21.51-.15 1.07-.14 1.61.24 1.64 1.82 3.02 3.5 2.87 1.12-.01 2.19-.66 2.77-1.61.19-.33.4-.67.41-1.06.1-1.79.06-3.57.07-5.36.01-4.03-.01-8.05.02-12.07z"/>
                                </svg>
                            </a>
                        <?php endif; ?>
                        
                        <?php if ($social_facebook): ?>
                            <a href="<?php echo esc_url($social_facebook); ?>" target="_blank" class="kdd-social-icon" title="Facebook">
                                <svg viewBox="0 0 24 24">
                                    <path d="M24 12.073c0-6.627-5.373-12-12-12s-12 5.373-12 12c0 5.99 4.388 10.954 10.125 11.854v-8.385H7.078v-3.47h3.047V9.43c0-3.007 1.792-4.669 4.533-4.669 1.312 0 2.686.235 2.686.235v2.953H15.83c-1.491 0-1.956.925-1.956 1.874v2.25h3.328l-.532 3.47h-2.796v8.385C19.612 23.027 24 18.062 24 12.073z"/>
                                </svg>
                            </a>
                        <?php endif; ?>
                        
                        <?php if ($social_twitter): ?>
                            <a href="<?php echo esc_url($social_twitter); ?>" target="_blank" class="kdd-social-icon" title="X (Twitter)">
                                <svg viewBox="0 0 24 24">
                                    <path d="M18.244 2.25h3.308l-7.227 8.26 8.502 11.24H16.17l-5.214-6.817L4.99 21.75H1.68l7.73-8.835L1.254 2.25H8.08l4.713 6.231zm-1.161 17.52h1.833L7.084 4.126H5.117z"/>
                                </svg>
                            </a>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
                
                <div class="kdd-footer">
                    <p><span class="kdd-site-name"><?php echo esc_html(get_bloginfo('name')); ?></span> &copy; <?php echo date('Y'); ?></p>
                </div>
            </div>

            <?php
            // NUEVO: Botón flotante de WhatsApp (v2.0)
            if ($whatsapp_enabled && $whatsapp_number):
            ?>
                <a href="https://wa.me/<?php echo esc_attr($whatsapp_number); ?>" 
                   target="_blank" 
                   class="kdd-whatsapp-float"
                   title="Chatear en WhatsApp">
                    <svg viewBox="0 0 24 24">
                        <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893"/>
                    </svg>
                    <div class="kdd-whatsapp-tooltip">Chatear en WhatsApp</div>
                </a>
            <?php endif; ?>

            <!-- Script para mejorar compatibilidad -->
            <script>
                // Asegurar que los videos se ajusten correctamente en escritorio
                document.addEventListener('DOMContentLoaded', function() {
                    if (!/Mobi|Android/i.test(navigator.userAgent)) {
                        var videoFrames = document.querySelectorAll('.kdd-video-background iframe');
                        
                        videoFrames.forEach(function(frame) {
                            frame.style.width = '100vw';
                            frame.style.height = '100vh';
                            frame.style.minWidth = '100%';
                            frame.style.minHeight = '100%';
                            frame.style.objectFit = 'cover';
                        });
                    }
                });
            </script>
        </body>
        </html>
        <?php
        exit;
    }
}

// Inicializar el plugin
new KDDMantenimiento();

// Hook de desinstalación
register_uninstall_hook(__FILE__, 'kdd_mantenimiento_uninstall');
function kdd_mantenimiento_uninstall() {
    delete_option('kdd_mantenimiento_settings');
    delete_option('kddm_version');
}