<?php
class KDDTormentaObjetosAdmin {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'settings_init'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_kdd_upload_custom_image', array($this, 'handle_image_upload'));
    }
    
    public function add_admin_menu() {
        add_options_page(
            'KDD Tormenta de Objetos',
            'KDD Tormenta de Objetos',
            'manage_options',
            'kdd-tormenta-objetos',
            array($this, 'options_page')
        );
    }
    
    public function settings_init() {
        register_setting(
            'kdd_tormenta_objetos', 
            'kdd_tormenta_objetos_settings',
            array($this, 'sanitize_settings')
        );
        
        add_settings_section(
            'kdd_tormenta_objetos_section',
            __('Configuración de Tormenta de Objetos', 'kdd-tormenta-objetos'),
            array($this, 'settings_section_callback'),
            'kdd_tormenta_objetos'
        );
        
        add_settings_field(
            'active',
            __('Activar efecto', 'kdd-tormenta-objetos'),
            array($this, 'active_render'),
            'kdd_tormenta_objetos',
            'kdd_tormenta_objetos_section'
        );
        
        add_settings_field(
            'object_type',
            __('Tipo de objeto', 'kdd-tormenta-objetos'),
            array($this, 'object_type_render'),
            'kdd_tormenta_objetos',
            'kdd_tormenta_objetos_section'
        );
        
        add_settings_field(
            'object_count',
            __('Cantidad de objetos', 'kdd-tormenta-objetos'),
            array($this, 'object_count_render'),
            'kdd_tormenta_objetos',
            'kdd_tormenta_objetos_section'
        );
        
        add_settings_field(
            'object_speed',
            __('Velocidad', 'kdd-tormenta-objetos'),
            array($this, 'object_speed_render'),
            'kdd_tormenta_objetos',
            'kdd_tormenta_objetos_section'
        );
        
        add_settings_field(
            'object_size',
            __('Tamaño de objetos', 'kdd-tormenta-objetos'),
            array($this, 'object_size_render'),
            'kdd_tormenta_objetos',
            'kdd_tormenta_objetos_section'
        );
        
        // Nuevos campos para el modo de fechas
        add_settings_field(
            'date_mode',
            __('Modo automático por fechas', 'kdd-tormenta-objetos'),
            array($this, 'date_mode_render'),
            'kdd_tormenta_objetos',
            'kdd_tormenta_objetos_section'
        );
        
        add_settings_field(
            'start_date',
            __('Fecha de inicio', 'kdd-tormenta-objetos'),
            array($this, 'start_date_render'),
            'kdd_tormenta_objetos',
            'kdd_tormenta_objetos_section'
        );
        
        add_settings_field(
            'end_date',
            __('Fecha de fin', 'kdd-tormenta-objetos'),
            array($this, 'end_date_render'),
            'kdd_tormenta_objetos',
            'kdd_tormenta_objetos_section'
        );
        
        // Nuevos campos para el modo temporizador
        add_settings_field(
            'timer_mode',
            __('Modo temporizador', 'kdd-tormenta-objetos'),
            array($this, 'timer_mode_render'),
            'kdd_tormenta_objetos',
            'kdd_tormenta_objetos_section'
        );
        
        add_settings_field(
            'timer_duration',
            __('Duración del temporizador', 'kdd-tormenta-objetos'),
            array($this, 'timer_duration_render'),
            'kdd_tormenta_objetos',
            'kdd_tormenta_objetos_section'
        );
    }
    
    public function sanitize_settings($input) {
        $sanitized_input = array();
        
        // Sanitizar active (checkbox)
        $sanitized_input['active'] = isset($input['active']) ? '1' : '0';
        
        // Sanitizar object_type
        $allowed_types = array(
            'snowflake', 'candy', 'star', 'sleigh', 'snowman', 'ball', 'bell',
            'bat', 'pumpkin', 'witchhat', 'spiderweb', 'skull', 'leaf', 'raindrop',
            'umbrella', 'heart', 'rose', 'turkey', 'balloon', 'sticker', 'custom'
        );
        $sanitized_input['object_type'] = isset($input['object_type']) && in_array($input['object_type'], $allowed_types) 
            ? sanitize_text_field($input['object_type']) 
            : 'snowflake';
        
        // Sanitizar object_count (número entre 1 y 100)
        $sanitized_input['object_count'] = isset($input['object_count']) 
            ? absint($input['object_count']) 
            : 30;
        $sanitized_input['object_count'] = max(1, min(100, $sanitized_input['object_count']));
        
        // Sanitizar object_speed (número entre 1 y 20)
        $sanitized_input['object_speed'] = isset($input['object_speed']) 
            ? absint($input['object_speed']) 
            : 5;
        $sanitized_input['object_speed'] = max(1, min(20, $sanitized_input['object_speed']));
        
        // ACTUALIZADO: Sanitizar object_size (número entre 5 y 100)
        $sanitized_input['object_size'] = isset($input['object_size']) 
            ? absint($input['object_size']) 
            : 15;
        $sanitized_input['object_size'] = max(5, min(100, $sanitized_input['object_size']));
        
        // Sanitizar date_mode (checkbox)
        $sanitized_input['date_mode'] = isset($input['date_mode']) ? '1' : '0';
        
        // Sanitizar start_date y end_date
        $sanitized_input['start_date'] = isset($input['start_date']) ? sanitize_text_field($input['start_date']) : '';
        $sanitized_input['end_date'] = isset($input['end_date']) ? sanitize_text_field($input['end_date']) : '';
        
        // Sanitizar timer_mode (checkbox)
        $sanitized_input['timer_mode'] = isset($input['timer_mode']) ? '1' : '0';
        
        // Sanitizar timer_duration (número entre 10 y 40)
        $sanitized_input['timer_duration'] = isset($input['timer_duration']) 
            ? absint($input['timer_duration']) 
            : 20;
        $sanitized_input['timer_duration'] = max(10, min(40, $sanitized_input['timer_duration']));
        
        return $sanitized_input;
    }
    
    public function active_render() {
        $options = get_option('kdd_tormenta_objetos_settings');
        ?>
        <input type='checkbox' name='kdd_tormenta_objetos_settings[active]' <?php checked(isset($options['active']) ? $options['active'] : 0, 1); ?> value='1'>
        <label for='kdd_tormenta_objetos_settings[active]'><?php esc_html_e('Activar el efecto de tormenta de objetos en el sitio', 'kdd-tormenta-objetos'); ?></label>
        <?php
    }
    
    public function object_type_render() {
        $options = get_option('kdd_tormenta_objetos_settings');
        $selected = isset($options['object_type']) ? $options['object_type'] : 'snowflake';
        ?>
        <select name='kdd_tormenta_objetos_settings[object_type]' id='object_type_select' style='width: 300px;'>
            <option value='snowflake' <?php selected($selected, 'snowflake'); ?>><?php esc_html_e('❄ Copos de nieve', 'kdd-tormenta-objetos'); ?></option>
            <option value='candy' <?php selected($selected, 'candy'); ?>><?php esc_html_e('🍬 Dulces', 'kdd-tormenta-objetos'); ?></option>
            <option value='star' <?php selected($selected, 'star'); ?>><?php esc_html_e('⭐ Estrellas', 'kdd-tormenta-objetos'); ?></option>
            <option value='sleigh' <?php selected($selected, 'sleigh'); ?>><?php esc_html_e('🛷 Trineos', 'kdd-tormenta-objetos'); ?></option>
            <option value='snowman' <?php selected($selected, 'snowman'); ?>><?php esc_html_e('⛄ Muñecos de nieve', 'kdd-tormenta-objetos'); ?></option>
            <option value='ball' <?php selected($selected, 'ball'); ?>><?php esc_html_e('⚽ Pelotas', 'kdd-tormenta-objetos'); ?></option>
            <option value='bell' <?php selected($selected, 'bell'); ?>><?php esc_html_e('🔔 Campanas navideñas', 'kdd-tormenta-objetos'); ?></option>
            <option value='bat' <?php selected($selected, 'bat'); ?>><?php esc_html_e('🦇 Murciélagos de Halloween', 'kdd-tormenta-objetos'); ?></option>
            <option value='pumpkin' <?php selected($selected, 'pumpkin'); ?>><?php esc_html_e('🎃 Calabazas de Halloween', 'kdd-tormenta-objetos'); ?></option>
            <option value='witchhat' <?php selected($selected, 'witchhat'); ?>><?php esc_html_e('🧙 Sombreros de brujas', 'kdd-tormenta-objetos'); ?></option>
            <option value='spiderweb' <?php selected($selected, 'spiderweb'); ?>><?php esc_html_e('🕸️ Telarañas de Halloween', 'kdd-tormenta-objetos'); ?></option>
            <option value='skull' <?php selected($selected, 'skull'); ?>><?php esc_html_e('💀 Cráneos de Halloween', 'kdd-tormenta-objetos'); ?></option>
            <option value='leaf' <?php selected($selected, 'leaf'); ?>><?php esc_html_e('🍂 Hojas de otoño/primavera', 'kdd-tormenta-objetos'); ?></option>
            <option value='raindrop' <?php selected($selected, 'raindrop'); ?>><?php esc_html_e('💧 Gotas de lluvia', 'kdd-tormenta-objetos'); ?></option>
            <option value='umbrella' <?php selected($selected, 'umbrella'); ?>><?php esc_html_e('☂ Paraguas', 'kdd-tormenta-objetos'); ?></option>
            <option value='heart' <?php selected($selected, 'heart'); ?>><?php esc_html_e('❤ Corazones de San Valentín', 'kdd-tormenta-objetos'); ?></option>
            <option value='rose' <?php selected($selected, 'rose'); ?>><?php esc_html_e('🌹 Rosas de San Valentín', 'kdd-tormenta-objetos'); ?></option>
            <option value='turkey' <?php selected($selected, 'turkey'); ?>><?php esc_html_e('🦃 Pavos de Acción de Gracias', 'kdd-tormenta-objetos'); ?></option>
            <option value='balloon' <?php selected($selected, 'balloon'); ?>><?php esc_html_e('🎈 Globos de Año Nuevo', 'kdd-tormenta-objetos'); ?></option>
            <option value='sticker' <?php selected($selected, 'sticker'); ?>><?php esc_html_e('🏷 Pegatinas', 'kdd-tormenta-objetos'); ?></option>
            <option value='custom' <?php selected($selected, 'custom'); ?>><?php esc_html_e('🖼️ Personalizado', 'kdd-tormenta-objetos'); ?></option>
        </select>
        <?php
    }
    
    public function object_count_render() {
        $options = get_option('kdd_tormenta_objetos_settings');
        $value = isset($options['object_count']) ? $options['object_count'] : 30;
        ?>
        <input type='range' name='kdd_tormenta_objetos_settings[object_count]' min='1' max='100' value='<?php echo esc_attr($value); ?>' oninput="this.nextElementSibling.value = this.value + ' objetos'">
        <output><?php echo esc_html($value); ?> objetos</output>
        <p class="description"><?php esc_html_e('Cantidad de objetos que caerán simultáneamente', 'kdd-tormenta-objetos'); ?></p>
        <?php
    }
    
    public function object_speed_render() {
        $options = get_option('kdd_tormenta_objetos_settings');
        $value = isset($options['object_speed']) ? $options['object_speed'] : 5;
        ?>
        <input type='range' name='kdd_tormenta_objetos_settings[object_speed]' min='1' max='20' value='<?php echo esc_attr($value); ?>' oninput="this.nextElementSibling.value = this.value">
        <output><?php echo esc_html($value); ?></output>
        <p class="description"><?php esc_html_e('Velocidad de caída de los objetos (1 = lento, 20 = rápido)', 'kdd-tormenta-objetos'); ?></p>
        <?php
    }
    
    public function object_size_render() {
        $options = get_option('kdd_tormenta_objetos_settings');
        $value = isset($options['object_size']) ? $options['object_size'] : 15;
        ?>
        <input type='range' name='kdd_tormenta_objetos_settings[object_size]' min='5' max='100' value='<?php echo esc_attr($value); ?>' oninput="this.nextElementSibling.value = this.value + 'px'">
        <output><?php echo esc_html($value); ?>px</output>
        <p class="description"><?php esc_html_e('Tamaño base de los objetos en píxeles', 'kdd-tormenta-objetos'); ?></p>
        <?php
    }
    
    public function date_mode_render() {
        $options = get_option('kdd_tormenta_objetos_settings');
        ?>
        <input type='checkbox' name='kdd_tormenta_objetos_settings[date_mode]' <?php checked(isset($options['date_mode']) ? $options['date_mode'] : 0, 1); ?> value='1'>
        <label for='kdd_tormenta_objetos_settings[date_mode]'><?php esc_html_e('Activar modo automático por fechas', 'kdd-tormenta-objetos'); ?></label>
        <p class="description"><?php esc_html_e('El efecto solo se activará entre las fechas especificadas', 'kdd-tormenta-objetos'); ?></p>
        <?php
    }
    
    public function start_date_render() {
        $options = get_option('kdd_tormenta_objetos_settings');
        $value = isset($options['start_date']) ? $options['start_date'] : '';
        ?>
        <input type='datetime-local' name='kdd_tormenta_objetos_settings[start_date]' value='<?php echo esc_attr($value); ?>'>
        <p class="description"><?php esc_html_e('Fecha y hora de inicio para activar el efecto', 'kdd-tormenta-objetos'); ?></p>
        <?php
    }
    
    public function end_date_render() {
        $options = get_option('kdd_tormenta_objetos_settings');
        $value = isset($options['end_date']) ? $options['end_date'] : '';
        ?>
        <input type='datetime-local' name='kdd_tormenta_objetos_settings[end_date]' value='<?php echo esc_attr($value); ?>'>
        <p class="description"><?php esc_html_e('Fecha y hora de fin para desactivar el efecto', 'kdd-tormenta-objetos'); ?></p>
        <?php
    }
    
    public function timer_mode_render() {
        $options = get_option('kdd_tormenta_objetos_settings');
        ?>
        <input type='checkbox' name='kdd_tormenta_objetos_settings[timer_mode]' <?php checked(isset($options['timer_mode']) ? $options['timer_mode'] : 0, 1); ?> value='1'>
        <label for='kdd_tormenta_objetos_settings[timer_mode]'><?php esc_html_e('Activar modo temporizador', 'kdd-tormenta-objetos'); ?></label>
        <p class="description"><?php esc_html_e('El efecto se desactivará automáticamente después del tiempo configurado', 'kdd-tormenta-objetos'); ?></p>
        <?php
    }
    
    public function timer_duration_render() {
        $options = get_option('kdd_tormenta_objetos_settings');
        $value = isset($options['timer_duration']) ? $options['timer_duration'] : 20;
        ?>
        <input type='range' name='kdd_tormenta_objetos_settings[timer_duration]' min='10' max='40' value='<?php echo esc_attr($value); ?>' oninput="this.nextElementSibling.value = this.value + ' segundos'">
        <output><?php echo esc_html($value); ?> segundos</output>
        <p class="description"><?php esc_html_e('Duración del efecto antes de desactivarse automáticamente', 'kdd-tormenta-objetos'); ?></p>
        <?php
    }
    
    public function settings_section_callback() {
        echo esc_html__('Configura los parámetros del efecto de tormenta de objetos en tu sitio web.', 'kdd-tormenta-objetos');
    }
    
    public function handle_image_upload() {
        // Verificar permisos
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('No tienes permisos para realizar esta acción.', 'kdd-tormenta-objetos'));
        }
        
        // Verificar nonce y existencia de variables POST
        if (!isset($_POST['nonce']) || !isset($_POST['object_type'])) {
            wp_die(esc_html__('Datos incompletos.', 'kdd-tormenta-objetos'));
        }
        
        // Sanitizar y verificar nonce
        $nonce = sanitize_text_field(wp_unslash($_POST['nonce']));
        if (!wp_verify_nonce($nonce, 'kdd_tormenta_objetos_upload')) {
            wp_die(esc_html__('Error de seguridad.', 'kdd-tormenta-objetos'));
        }
        
        // Sanitizar object_type
        $object_type = sanitize_text_field(wp_unslash($_POST['object_type']));
        
        // Verificar que se haya subido un archivo
        if (!isset($_FILES['custom_image']) || empty($_FILES['custom_image']['name'])) {
            wp_send_json_error(esc_html__('No se recibió ninguna imagen.', 'kdd-tormenta-objetos'));
        }
        
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        
        $upload = wp_handle_upload($_FILES['custom_image'], array('test_form' => false));
        
        if (isset($upload['error'])) {
            wp_send_json_error($upload['error']);
        }
        
        $images = get_option('kdd_tormenta_objetos_images', array());
        $images[$object_type] = $upload['url'];
        update_option('kdd_tormenta_objetos_images', $images);
        
        wp_send_json_success(array(
            'url' => esc_url($upload['url']),
            'message' => esc_html__('Imagen subida correctamente.', 'kdd-tormenta-objetos')
        ));
    }
    
    public function options_page() {
        $images = get_option('kdd_tormenta_objetos_images', array());
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('KDD Tormenta de Objetos - Configuración', 'kdd-tormenta-objetos'); ?></h1>
            
            <div style="display: flex; gap: 20px; margin-top: 20px;">
                <div style="flex: 2;">
                    <form action='options.php' method='post'>
                        <?php
                        settings_fields('kdd_tormenta_objetos');
                        do_settings_sections('kdd_tormenta_objetos');
                        submit_button(esc_html__('Guardar Configuración', 'kdd-tormenta-objetos'));
                        ?>
                    </form>
                    
                    <hr style="margin: 30px 0;">
                    
                    <h2><?php esc_html_e('Imágenes Personalizadas', 'kdd-tormenta-objetos'); ?></h2>
                    <p><?php esc_html_e('Sube tus propias imágenes PNG para los objetos. Tamaño recomendado: 100x100 píxeles.', 'kdd-tormenta-objetos'); ?></p>
                    
                    <div id="custom-images-section">
                        <div style="margin-bottom: 15px;">
                            <label for="image_object_type"><strong><?php esc_html_e('Seleccionar tipo de objeto:', 'kdd-tormenta-objetos'); ?></strong></label>
                            <select id="image_object_type" style="margin-left: 10px;">
                                <option value="snowflake">❄ <?php esc_html_e('Copos de nieve', 'kdd-tormenta-objetos'); ?></option>
                                <option value="candy">🍬 <?php esc_html_e('Dulces', 'kdd-tormenta-objetos'); ?></option>
                                <option value="star">⭐ <?php esc_html_e('Estrellas', 'kdd-tormenta-objetos'); ?></option>
                                <option value="sleigh">🛷 <?php esc_html_e('Trineos', 'kdd-tormenta-objetos'); ?></option>
                                <option value="snowman">⛄ <?php esc_html_e('Muñecos de nieve', 'kdd-tormenta-objetos'); ?></option>
                                <option value="ball">⚽ <?php esc_html_e('Pelotas', 'kdd-tormenta-objetos'); ?></option>
                                <option value="bell">🔔 <?php esc_html_e('Campanas navideñas', 'kdd-tormenta-objetos'); ?></option>
                                <option value="bat">🦇 <?php esc_html_e('Murciélagos', 'kdd-tormenta-objetos'); ?></option>
                                <option value="pumpkin">🎃 <?php esc_html_e('Calabazas', 'kdd-tormenta-objetos'); ?></option>
                                <option value="witchhat">🧙 <?php esc_html_e('Sombreros de brujas', 'kdd-tormenta-objetos'); ?></option>
                                <option value="spiderweb">🕸️ <?php esc_html_e('Telarañas', 'kdd-tormenta-objetos'); ?></option>
                                <option value="skull">💀 <?php esc_html_e('Cráneos', 'kdd-tormenta-objetos'); ?></option>
                                <option value="leaf">🍂 <?php esc_html_e('Hojas', 'kdd-tormenta-objetos'); ?></option>
                                <option value="raindrop">💧 <?php esc_html_e('Gotas de lluvia', 'kdd-tormenta-objetos'); ?></option>
                                <option value="umbrella">☂ <?php esc_html_e('Paraguas', 'kdd-tormenta-objetos'); ?></option>
                                <option value="heart">❤ <?php esc_html_e('Corazones', 'kdd-tormenta-objetos'); ?></option>
                                <option value="rose">🌹 <?php esc_html_e('Rosas', 'kdd-tormenta-objetos'); ?></option>
                                <option value="turkey">🦃 <?php esc_html_e('Pavos', 'kdd-tormenta-objetos'); ?></option>
                                <option value="balloon">🎈 <?php esc_html_e('Globos', 'kdd-tormenta-objetos'); ?></option>
                                <option value="sticker">🏷 <?php esc_html_e('Pegatinas', 'kdd-tormenta-objetos'); ?></option>
                                <option value="custom">🖼️ <?php esc_html_e('Personalizado', 'kdd-tormenta-objetos'); ?></option>
                            </select>
                        </div>
                        
                        <div style="margin-bottom: 15px;">
                            <input type="file" id="custom_image_upload" accept=".png,.jpg,.jpeg,.gif" style="margin-bottom: 10px;">
                            <button type="button" id="upload_image_btn" class="button"><?php esc_html_e('Subir Imagen', 'kdd-tormenta-objetos'); ?></button>
                        </div>
                        
                        <div id="upload_status" style="margin-bottom: 15px;"></div>
                        
                        <div id="current_images">
                            <h3><?php esc_html_e('Imágenes Actuales', 'kdd-tormenta-objetos'); ?></h3>
                            <?php if (empty($images)): ?>
                                <p><?php esc_html_e('No hay imágenes personalizadas subidas.', 'kdd-tormenta-objetos'); ?></p>
                            <?php else: ?>
                                <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(150px, 1fr)); gap: 15px;">
                                    <?php foreach ($images as $type => $url): ?>
                                        <div style="border: 1px solid #ddd; padding: 10px; text-align: center;">
                                            <img src="<?php echo esc_url($url); ?>" style="max-width: 50px; max-height: 50px; margin-bottom: 5px;">
                                            <div style="font-size: 12px; word-break: break-all;"><?php echo esc_html($type); ?></div>
                                            <button type="button" class="button button-small remove-image" data-type="<?php echo esc_attr($type); ?>" style="margin-top: 5px;"><?php esc_html_e('Eliminar', 'kdd-tormenta-objetos'); ?></button>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                
                <div style="flex: 1; background: #fff; padding: 20px; border-radius: 5px; border: 1px solid #ccd0d4;">
                    <h3><?php esc_html_e('Vista previa', 'kdd-tormenta-objetos'); ?></h3>
                    <div id="kdd-tormenta-objetos-preview" style="height: 200px; border: 1px dashed #ccc; position: relative; overflow: hidden; background: #f8f9fa; margin-bottom: 15px;"></div>
                    <p><small><?php esc_html_e('La vista previa muestra la configuración actual. El efecto real se verá en tu sitio web.', 'kdd-tormenta-objetos'); ?></small></p>
                    
                    <h3><?php esc_html_e('Tipos de Objetos', 'kdd-tormenta-objetos'); ?></h3>
                    <ul style="columns: 2; font-size: 12px;">
                        <li>❄ <?php esc_html_e('Copos de nieve', 'kdd-tormenta-objetos'); ?></li>
                        <li>🍬 <?php esc_html_e('Dulces', 'kdd-tormenta-objetos'); ?></li>
                        <li>⭐ <?php esc_html_e('Estrellas', 'kdd-tormenta-objetos'); ?></li>
                        <li>🛷 <?php esc_html_e('Trineos', 'kdd-tormenta-objetos'); ?></li>
                        <li>⛄ <?php esc_html_e('Muñecos de nieve', 'kdd-tormenta-objetos'); ?></li>
                        <li>⚽ <?php esc_html_e('Pelotas', 'kdd-tormenta-objetos'); ?></li>
                        <li>🔔 <?php esc_html_e('Campanas navideñas', 'kdd-tormenta-objetos'); ?></li>
                        <li>🦇 <?php esc_html_e('Murciélagos', 'kdd-tormenta-objetos'); ?></li>
                        <li>🎃 <?php esc_html_e('Calabazas', 'kdd-tormenta-objetos'); ?></li>
                        <li>🧙 <?php esc_html_e('Sombreros de brujas', 'kdd-tormenta-objetos'); ?></li>
                        <li>🕸️ <?php esc_html_e('Telarañas', 'kdd-tormenta-objetos'); ?></li>
                        <li>💀 <?php esc_html_e('Cráneos', 'kdd-tormenta-objetos'); ?></li>
                        <li>🍂 <?php esc_html_e('Hojas', 'kdd-tormenta-objetos'); ?></li>
                        <li>💧 <?php esc_html_e('Gotas de lluvia', 'kdd-tormenta-objetos'); ?></li>
                        <li>☂ <?php esc_html_e('Paraguas', 'kdd-tormenta-objetos'); ?></li>
                        <li>❤ <?php esc_html_e('Corazones', 'kdd-tormenta-objetos'); ?></li>
                        <li>🌹 <?php esc_html_e('Rosas', 'kdd-tormenta-objetos'); ?></li>
                        <li>🦃 <?php esc_html_e('Pavos', 'kdd-tormenta-objetos'); ?></li>
                        <li>🎈 <?php esc_html_e('Globos', 'kdd-tormenta-objetos'); ?></li>
                        <li>🏷 <?php esc_html_e('Pegatinas', 'kdd-tormenta-objetos'); ?></li>
                        <li>🖼️ <?php esc_html_e('Personalizado', 'kdd-tormenta-objetos'); ?></li>
                    </ul>
                </div>
            </div>
        </div>
        <?php
    }
    
    public function enqueue_admin_scripts($hook) {
        if ('settings_page_kdd-tormenta-objetos' !== $hook) {
            return;
        }
        
        wp_enqueue_script('kdd-tormenta-objetos-admin', KDD_TORMENTA_OBJETOS_PLUGIN_URL . 'admin/admin-scripts.js', array('jquery'), KDD_TORMENTA_OBJETOS_VERSION, true);
        wp_enqueue_style('kdd-tormenta-objetos-admin-css', KDD_TORMENTA_OBJETOS_PLUGIN_URL . 'assets/falling-objects.css', array(), KDD_TORMENTA_OBJETOS_VERSION);
        
        wp_localize_script('kdd-tormenta-objetos-admin', 'kddTormentaObjetosAdmin', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('kdd_tormenta_objetos_upload'),
            'uploading_text' => esc_html__('Subiendo imagen...', 'kdd-tormenta-objetos'),
            'success_text' => esc_html__('Imagen subida correctamente.', 'kdd-tormenta-objetos'),
            'error_text' => esc_html__('Error al subir la imagen.', 'kdd-tormenta-objetos'),
            'confirm_remove' => esc_html__('¿Estás seguro de que quieres eliminar esta imagen?', 'kdd-tormenta-objetos')
        ));
    }
}
?>