(function() {
    'use strict';
    
    let fallingObjects = [];
    let animationId = null;
    let isStopping = false;
    let timerStart = null;
    
    function initFallingObjects() {
        // Verificar si hay configuraciones
        if (typeof kddTormentaObjetosSettings === 'undefined') {
            return;
        }
        
        // Limpiar objetos existentes
        cleanup();
        
        // Resetear estado
        isStopping = false;
        timerStart = Date.now();
        
        // Crear nuevos objetos
        createObjects();
        
        // Iniciar animación
        animate();
    }
    
    function createObjects() {
        const container = document.getElementById('kdd-tormenta-objetos-container');
        if (!container) return;
        
        // CORRECCIÓN: Crear EXACTAMENTE la cantidad configurada
        const objectCount = parseInt(kddTormentaObjetosSettings.count);
        console.log('Creando objetos:', objectCount); // Para debug
        
        for (let i = 0; i < objectCount; i++) {
            createObject(container, i);
        }
        
        console.log('Objetos creados:', fallingObjects.length); // Para debug
    }
    
    function createObject(container, index) {
        const object = document.createElement('div');
        object.className = 'kdd-falling-object';
        object.dataset.index = index;
        
        // Tamaño basado en configuración con variación aleatoria
        const baseSize = parseInt(kddTormentaObjetosSettings.size);
        const size = baseSize + (Math.random() * baseSize * 0.3); // Menor variación
        
        // CORRECCIÓN MEJORADA: Distribución uniforme en TODA la pantalla
        // Usar el ancho completo sin márgenes
        const startX = Math.random() * (window.innerWidth + 200) - 100; // Extender 100px más allá de los bordes
        
        // Configurar objeto según el tipo
        configureObject(object, kddTormentaObjetosSettings.type, size);
        
        // Estilos
        object.style.width = `${size}px`;
        object.style.height = `${size}px`;
        object.style.left = `${startX}px`;
        object.style.top = `-${size}px`;
        object.style.opacity = '1';
        object.style.transition = 'opacity 2s ease-out';
        
        // CORRECCIÓN COMPLETA: Velocidad correcta
        const baseSpeed = parseInt(kddTormentaObjetosSettings.speed);
        // Convertir 1-20 a un rango de velocidad adecuado (1=lento, 20=rápido)
        const speedMultiplier = 0.3 + (baseSpeed * 0.15); // 1→0.45, 20→3.3
        
        const objData = {
            element: object,
            x: startX,
            y: -size,
            initialX: startX,
            // CORRECCIÓN: Velocidad CORRECTA - valores más altos = más rápido
            speed: (Math.random() * 1.5 + 0.5) * speedMultiplier,
            // Amplitud balanceada para movimiento natural
            amplitude: (Math.random() - 0.5) * 60, // -30 a +30
            angle: Math.random() * 360,
            rotationSpeed: (Math.random() - 0.5) * 2,
            size: size,
            swing: Math.random() * Math.PI * 2,
            swingSpeed: 0.008 + Math.random() * 0.015
        };
        
        fallingObjects.push(objData);
        container.appendChild(object);
    }
    
    function configureObject(object, type, size) {
        // Primero verificar si hay una imagen personalizada
        if (kddTormentaObjetosSettings.images && kddTormentaObjetosSettings.images[type]) {
            // Usar imagen personalizada
            object.innerHTML = `<img src="${kddTormentaObjetosSettings.images[type]}" alt="${type}" style="width:100%;height:100%;object-fit:contain;">`;
        } else {
            // Usar SVG o emoji como fallback
            configureFallbackObject(object, type, size);
        }
    }
    
    function configureFallbackObject(object, type, size) {
        const svgContent = getSVGForType(type);
        if (svgContent) {
            object.innerHTML = svgContent;
            const svg = object.querySelector('svg');
            if (svg) {
                svg.style.width = '100%';
                svg.style.height = '100%';
            }
        } else {
            // Fallback a emoji si no hay SVG
            object.innerHTML = getEmojiForType(type);
            object.style.fontSize = `${size * 0.8}px`; // Emoji un poco más pequeño
            object.style.textAlign = 'center';
            object.style.lineHeight = '1';
            object.style.display = 'flex';
            object.style.alignItems = 'center';
            object.style.justifyContent = 'center';
        }
    }
    
    function getSVGForType(type) {
        const svgs = {
            snowflake: `<svg viewBox="0 0 100 100" xmlns="http://www.w3.org/2000/svg">
                <path d="M50 15 L50 85 M30 30 L70 70 M70 30 L30 70 M15 50 L85 50 M25 25 L75 75 M75 25 L25 75" 
                      stroke="#E0F7FA" stroke-width="3" fill="none"/>
            </svg>`,
            
            candy: `<svg viewBox="0 0 100 100" xmlns="http://www.w3.org/2000/svg">
                <rect x="30" y="30" width="40" height="40" rx="5" fill="#FF4081"/>
                <path d="M30 35 L70 35 M30 45 L70 45 M30 55 L70 55 M30 65 L70 65" 
                      stroke="#FFFFFF" stroke-width="2"/>
            </svg>`,
            
            star: `<svg viewBox="0 0 100 100" xmlns="http://www.w3.org/2000/svg">
                <polygon points="50,15 61,40 88,40 66,57 72,85 50,70 28,85 34,57 12,40 39,40" 
                        fill="#FFEB3B" stroke="#FBC02D" stroke-width="2"/>
            </svg>`,
            
            heart: `<svg viewBox="0 0 100 100" xmlns="http://www.w3.org/2000/svg">
                <path d="M50,85 C30,65 15,50 15,30 C15,15 30,15 50,30 C70,15 85,15 85,30 C85,50 70,65 50,85 Z" 
                      fill="#F44336"/>
            </svg>`
        };
        return svgs[type] || null;
    }
    
    function getEmojiForType(type) {
        const emojis = {
            snowflake: '❄',
            candy: '🍬',
            star: '⭐',
            sleigh: '🛷',
            snowman: '⛄',
            ball: '⚽',
            bell: '🔔',
            bat: '🦇',
            pumpkin: '🎃',
            witchhat: '🧙',
            spiderweb: '🕸️',
            skull: '💀',
            leaf: '🍂',
            raindrop: '💧',
            umbrella: '☂',
            heart: '❤',
            rose: '🌹',
            turkey: '🦃',
            balloon: '🎈',
            sticker: '🏷',
            custom: '🖼️'
        };
        return emojis[type] || '❄';
    }
    
    function animate() {
        if (isStopping) return;
        
        const currentTime = Date.now();
        
        // Verificar temporizador si está activo
        if (kddTormentaObjetosSettings.timer_mode === '1' && kddTormentaObjetosSettings.timer_duration > 0) {
            const elapsedSeconds = (currentTime - timerStart) / 1000;
            const remainingTime = kddTormentaObjetosSettings.timer_duration - elapsedSeconds;
            
            // Iniciar desvanecimiento cuando queden 3 segundos
            if (remainingTime <= 3 && remainingTime > 0) {
                const fadeOpacity = remainingTime / 3; // De 1 a 0 en 3 segundos
                fallingObjects.forEach(obj => {
                    obj.element.style.opacity = fadeOpacity;
                });
            }
            
            // Detener completamente cuando se acabe el tiempo
            if (remainingTime <= 0) {
                stopAnimation();
                return;
            }
        }
        
        for (let i = 0; i < fallingObjects.length; i++) {
            const obj = fallingObjects[i];
            
            // CORRECCIÓN: Mover hacia abajo con velocidad correcta
            obj.y += obj.speed;
            
            // Movimiento horizontal balanceado
            obj.swing += obj.swingSpeed;
            const swingOffset = Math.sin(obj.swing) * obj.amplitude;
            obj.x = obj.initialX + swingOffset;
            
            // Rotación
            obj.angle += obj.rotationSpeed;
            
            // Si el objeto sale de la pantalla por abajo, reiniciarlo arriba
            if (obj.y > window.innerHeight) {
                obj.y = -obj.size;
                // CORRECCIÓN: Nueva posición X aleatoria en TODA la pantalla
                obj.initialX = Math.random() * (window.innerWidth + 200) - 100;
                obj.x = obj.initialX;
                obj.swing = Math.random() * Math.PI * 2;
            }
            
            // Si el objeto sale por los lados, hacerlo reaparecer del otro lado
            if (obj.x < -obj.size - 50) {
                obj.x = window.innerWidth + obj.size;
                obj.initialX = obj.x;
            }
            if (obj.x > window.innerWidth + obj.size + 50) {
                obj.x = -obj.size;
                obj.initialX = obj.x;
            }
            
            // Aplicar transformación
            obj.element.style.transform = `translate(${obj.x}px, ${obj.y}px) rotate(${obj.angle}deg)`;
        }
        
        animationId = requestAnimationFrame(animate);
    }
    
    function stopAnimation() {
        if (isStopping) return;
        isStopping = true;
        
        // Efecto de desvanecimiento final
        fallingObjects.forEach(obj => {
            obj.element.style.opacity = '0';
        });
        
        // Esperar a que termine la transición antes de limpiar
        setTimeout(() => {
            cleanup();
        }, 2000);
    }
    
    function cleanup() {
        if (animationId) {
            cancelAnimationFrame(animationId);
            animationId = null;
        }
        
        const container = document.getElementById('kdd-tormenta-objetos-container');
        if (container) {
            container.innerHTML = '';
        }
        
        fallingObjects = [];
        isStopping = false;
    }
    
    // Reiniciar cuando cambie el tamaño de la ventana
    let resizeTimeout;
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(function() {
            if (fallingObjects.length > 0 && !isStopping) {
                // Reajustar posiciones basadas en el nuevo tamaño
                for (let i = 0; i < fallingObjects.length; i++) {
                    const obj = fallingObjects[i];
                    if (obj.x > window.innerWidth) {
                        obj.x = window.innerWidth - 50;
                        obj.initialX = obj.x;
                    }
                }
            }
        }, 250);
    });
    
    // Inicializar cuando el DOM esté listo
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initFallingObjects);
    } else {
        initFallingObjects();
    }
    
    // Hacer disponible globalmente para posibles reinicios
    window.restartKDDTormentaObjetos = initFallingObjects;
})();