<?php
/**
 * Plugin Name: KDD Tormenta de Objetos
 * Plugin URI: https://kddbusiness.com
 * Description: Agrega un efecto de objetos cayendo a tu sitio web con diferentes temas estacionales.
 * Version: 1.1
 * Author: KDD Business
 * License: GPL v2 or later
 * Text Domain: kdd-tormenta-objetos
 * Requires at least: 5.0
 * Tested up to: 6.3
 * Requires PHP: 7.4
 */

// Evitar acceso directo
if (!defined('ABSPATH')) {
    exit;
}

// Definir constantes
define('KDD_TORMENTA_OBJETOS_VERSION', '1.1.0');
define('KDD_TORMENTA_OBJETOS_PLUGIN_URL', plugin_dir_url(__FILE__));
define('KDD_TORMENTA_OBJETOS_PLUGIN_PATH', plugin_dir_path(__FILE__));

// Inicializar el plugin
class KDDTormentaObjetosPlugin {
    
    public function __construct() {
        add_action('init', array($this, 'init'));
        add_action('wp_footer', array($this, 'display_falling_objects'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        add_action('wp_ajax_kdd_remove_custom_image', array($this, 'handle_image_removal'));
    }
    
    public function activate() {
        // Configuración por defecto al activar
        $default_options = array(
            'active' => '0',
            'object_type' => 'snowflake',
            'object_count' => '30',
            'object_speed' => '5',
            'object_size' => '15',
            'date_mode' => '0',
            'start_date' => '',
            'end_date' => '',
            'timer_mode' => '0',
            'timer_duration' => '20'
        );
        
        if (!get_option('kdd_tormenta_objetos_settings')) {
            add_option('kdd_tormenta_objetos_settings', $default_options);
        }
        
        // Crear directorio para imágenes personalizadas
        $upload_dir = wp_upload_dir();
        $custom_dir = $upload_dir['basedir'] . '/kdd-tormenta-objetos';
        if (!file_exists($custom_dir)) {
            wp_mkdir_p($custom_dir);
        }
    }
    
    public function deactivate() {
        // Limpiar si es necesario
    }
    
    public function init() {
        // Cargar archivos de administración
        if (is_admin()) {
            require_once KDD_TORMENTA_OBJETOS_PLUGIN_PATH . 'admin/admin-page.php';
            new KDDTormentaObjetosAdmin();
        }
        
        // Las traducciones se cargan automáticamente desde WordPress 4.6
    }
    
    public function display_falling_objects() {
        $options = get_option('kdd_tormenta_objetos_settings');
        
        // Verificar si está activo
        if (!isset($options['active']) || $options['active'] !== '1') {
            return;
        }
        
        // Verificar modo fechas si está activado
        if (isset($options['date_mode']) && $options['date_mode'] === '1') {
            if (!$this->is_within_date_range($options)) {
                return;
            }
        }
        
        // Encolar scripts y estilos
        wp_enqueue_script('kdd-tormenta-objetos-js', KDD_TORMENTA_OBJETOS_PLUGIN_URL . 'assets/falling-objects.js', array(), KDD_TORMENTA_OBJETOS_VERSION, true);
        wp_enqueue_style('kdd-tormenta-objetos-css', KDD_TORMENTA_OBJETOS_PLUGIN_URL . 'assets/falling-objects.css', array(), KDD_TORMENTA_OBJETOS_VERSION);
        
        // Pasar opciones a JavaScript
        $object_images = get_option('kdd_tormenta_objetos_images', array());
        
        // CORRECCIÓN: Velocidad directa sin invertir
        $js_speed = isset($options['object_speed']) ? intval($options['object_speed']) : 5;
        
        // Configurar temporizador
        $timer_duration = 0;
        if (isset($options['timer_mode']) && $options['timer_mode'] === '1') {
            $timer_duration = isset($options['timer_duration']) ? intval($options['timer_duration']) : 20;
        }
        
        wp_localize_script('kdd-tormenta-objetos-js', 'kddTormentaObjetosSettings', array(
            'type' => isset($options['object_type']) ? $options['object_type'] : 'snowflake',
            'count' => isset($options['object_count']) ? intval($options['object_count']) : 30,
            'speed' => $js_speed,
            'size' => isset($options['object_size']) ? intval($options['object_size']) : 15,
            'active' => isset($options['active']) ? $options['active'] : '0',
            'images' => $object_images,
            'timer_mode' => isset($options['timer_mode']) ? $options['timer_mode'] : '0',
            'timer_duration' => $timer_duration
        ));
        
        // Contenedor para los objetos
        echo '<div id="kdd-tormenta-objetos-container"></div>';
    }
    
    /**
     * Verifica si la fecha actual está dentro del rango configurado
     */
    private function is_within_date_range($options) {
        $current_time = current_time('timestamp');
        $start_date = isset($options['start_date']) ? strtotime($options['start_date']) : 0;
        $end_date = isset($options['end_date']) ? strtotime($options['end_date']) : 0;
        
        // Si no hay fechas configuradas, no activar
        if (!$start_date || !$end_date) {
            return false;
        }
        
        // Verificar si estamos dentro del rango
        return ($current_time >= $start_date && $current_time <= $end_date);
    }
    
    public function handle_image_removal() {
        // Verificar permisos
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('No tienes permisos para realizar esta acción.', 'kdd-tormenta-objetos'));
        }
        
        // Verificar nonce y existencia de variables POST
        if (!isset($_POST['nonce']) || !isset($_POST['object_type'])) {
            wp_die(esc_html__('Datos incompletos.', 'kdd-tormenta-objetos'));
        }
        
        // Sanitizar y verificar nonce
        $nonce = sanitize_text_field(wp_unslash($_POST['nonce']));
        if (!wp_verify_nonce($nonce, 'kdd_tormenta_objetos_upload')) {
            wp_die(esc_html__('Error de seguridad.', 'kdd-tormenta-objetos'));
        }
        
        // Sanitizar object_type
        $object_type = sanitize_text_field(wp_unslash($_POST['object_type']));
        $images = get_option('kdd_tormenta_objetos_images', array());
        
        if (isset($images[$object_type])) {
            unset($images[$object_type]);
            update_option('kdd_tormenta_objetos_images', $images);
            wp_send_json_success(esc_html__('Imagen eliminada correctamente.', 'kdd-tormenta-objetos'));
        }
        
        wp_send_json_error(esc_html__('No se encontró la imagen.', 'kdd-tormenta-objetos'));
    }
}

// Inicializar el plugin
new KDDTormentaObjetosPlugin();

// Función para obtener la URL de una imagen personalizada
function kdd_tormenta_objetos_get_image_url($object_type) {
    $images = get_option('kdd_tormenta_objetos_images', array());
    if (isset($images[$object_type]) && !empty($images[$object_type])) {
        return $images[$object_type];
    }
    return false;
}
?>